﻿using System;
using System.Diagnostics;
using System.Net;
using System.Reflection;
using FyndSharp.Utilities.Common;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Protocols;

namespace gov.va.med.VBECS.Communication.Channels
{
#if NUNIT
    public
#else
    internal 
#endif      
    abstract class BaseChannel : IChannel
    {
        private readonly IPEndPoint _endPoint;

        protected BaseChannel()
        {
            _lockObject = new Object();
            Status = CommunicationStatus.Disconnected;
            LastReceivedTime = DateTime.MinValue;
            LastSentTime = DateTime.MinValue;
        }

        protected BaseChannel(IPEndPoint theEndPoint)
            : this()
        {
            Checker.NotNull(theEndPoint);
            _endPoint = new IPEndPoint(theEndPoint.Address, theEndPoint.Port);
        }

        // Logger object
        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        private readonly Object _lockObject;

        #region IChannel Members

        public event EventHandler<MessageEventArgs> MessageReceived;

        public event EventHandler<MessageEventArgs> MessageSent;

        public event EventHandler Disconnected;

        public DateTime LastReceivedTime { get; protected set; }

        public DateTime LastSentTime { get; protected set; }

        public IProtocol Protocol { get; set; }

        public IPEndPoint EndPoint
        {
            get { return _endPoint; }
        }

        public CommunicationStatus Status { get; protected set; }

        public void Send(IMessage aMessage)
        {
            Checker.NotNull(aMessage);
            SendImpl(aMessage);
            LastSentTime = DateTime.Now;
            FireMessageSentEvent(aMessage);
        }

        public void Start()
        {
            StartImpl();
            Status = CommunicationStatus.Connected;
        }

        public void Disconnect()
        {
            // if one disconnect already started we don't wont race conditions with other, it causes multiple FireDisconnectedEvents
            // make sure that Disconnect is not called from FireDisconnectedEvent!
            lock (_lockObject)
            {
                try
                {
                    DisconnectImpl();
                    FireDisconnectedEvent();
                }
#if TRACE
                catch (Exception e)
                {
                    Trace.WriteLine(e.ToString());
                }
#else
                catch { }
#endif
                Status = CommunicationStatus.Disconnected;
            }
        }

        #endregion

        protected abstract void SendImpl(IMessage aMessage);
        protected abstract void StartImpl();

        protected abstract void DisconnectImpl();

        protected virtual void FireDisconnectedEvent()
        {
            if (null != Disconnected)
            {
                Disconnected.Invoke(this, EventArgs.Empty);
            }
        }

        protected virtual void FireMessageSentEvent(IMessage theMessage)
        {
            if (null != MessageSent)
            {
                MessageSent.Invoke(this, new MessageEventArgs(theMessage));
            }
        }

        protected virtual void FireMessageReceivedEvent(IMessage theMessage)
        {
            if (null != MessageReceived)
            {
                MessageReceived.Invoke(this, new MessageEventArgs(theMessage));
            }

            // TODO: ultimately this code has to be uncommented and
            // if (message is ICloseConnectionResponceMessage)
            //     _isRunning = false;
            // removed from TcpChannel class
            // and Client.Disconnect should be removed after notify_vista_about_disconnect in VistALinkClientConnection
            // Basically, commented code disconnects channel 
            // Problem is that Disconnect event happens in different thread and statusBarPanelVistALink.Parent.Invoke hangs
            // I think it happens because modal view is opened at this time and we are trying to update Parent view which is disabled.

            //if (theMessage is ICloseConnectionResponceMessage)
            //{
            //    _logger.Debug("Close connection message received. Disconnect channel.");
            //    Disconnect();
            //}
        }
    }
}